<?php

namespace App\Repositories\General;

use Exception;
use Throwable;
use App\Models\Post;
use App\Models\Comment;
use App\Enums\SettingKey;
use Illuminate\Http\Request;
use App\Services\ImageService;
use App\Enums\NotificationType;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class CommentRepository
{
    protected $imageService;
    protected $notificationRepository;
    protected $postRepository;

    public function __construct(
        ImageService $imageService,
        NotificationRepository $notificationRepository,
        PostRepository $postRepository,
    ) {
        $this->imageService = $imageService;
        $this->notificationRepository = $notificationRepository;
        $this->postRepository = $postRepository;
    }

    public function save(Request $request)
    {
        try {
            DB::beginTransaction();
            $loginUser = auth()->user();

            $post = Post::findOrFail($request->post_id);
            $comment = Comment::updateOrCreate(
                ['id' => $request->id],
                [
                    'user_id' => $loginUser->id,
                    'parent_id' => $request->parent_id,
                    'post_id' => $post->id,
                    'content' => $request->content,
                ],
            );

            if (count($request->images) > 0) {
                $oldImages = $comment->images->pluck('id');
                if ($oldImages) {
                    $this->imageService->deleteMany($oldImages, Comment::class);
                }
                $images = $this->imageService->uploadMany($request->images, Comment::class, $comment->id);
                $comment->images()->saveMany($images);
            }

            if ($request->id == null && (bool) SettingRepository::getValueByName(SettingKey::NOTIFICATION_MY_POST_COMMENT, $post->user_id)) {
                $this->notificationRepository->add(
                    userId: $post->user_id,
                    content: $loginUser->name . ' commented on your post.',
                    imageUrl: $loginUser->image?->url,
                    type: NotificationType::POST->value,
                    redirectLink: route('post.show', $post->id)
                );
            }

            DB::commit();
        } catch (Throwable $e) {
            DB::rollBack();

            Log::error(__CLASS__ . '::' . __FUNCTION__ . '[line: ' . __LINE__ . ']Message: ' . $e->getMessage());

            throw new Exception('Comment saved failed.');
        }
    }

    public function destroy(Comment $comment)
    {
        abort_unless($comment->user_id == auth()->id(), 403, 'Unauthorized action.');

        $comment->delete();
    }

    public function like(Comment $comment)
    {
        $loginUser = auth()->user();

        try {
            DB::beginTransaction();

            $user = $comment->likes()->find($loginUser->id);
            if (!$user) {
                if ((bool) SettingRepository::getValueByName(SettingKey::NOTIFICATION_MY_POST_COMMENT, $comment->user_id)) {
                    $this->notificationRepository->add(
                        userId: $comment->user_id,
                        content: $loginUser->name . ' liked your comment.',
                        imageUrl: $loginUser->image?->url,
                        type: NotificationType::POST->value,
                        redirectLink: route('post.show', $comment->post_id)
                    );
                }
            }
            $comment->likes()->toggle([$loginUser->id]);

            DB::commit();
        } catch (Throwable $e) {
            DB::rollBack();

            Log::error(__CLASS__ . '::' . __FUNCTION__ . '[line: ' . __LINE__ . ']Message: ' . $e->getMessage());

            throw new Exception('Comment liked failed.');
        }

        return $this->postRepository->show($comment->post_id);
    }
}
